<?php
/**
 * Plugin Name: SBA 504 Loan Calculator
 * Description: Analyze SBA 504 Loans - Bank loan and CDC loan. It calculates : Monthly Payment, Total Cost of the loan, and Business Payment Schedule for the entire term (Principal | Interest Breakdown) with Visualization for both the Bank loan-1 and CDC loan-2. It provides [sba_504_loan_calculator] shortcode that renders the webcomponent.
 * Version: 1.0.0
 * Author: Simplifying Calculation
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Register JS + CSS assets (registered early, enqueued when shortcode used).
 * Also logs file presence to debug.log when WP_DEBUG is enabled.
 */
function msw_register_assets() {
    $plugin_url  = plugin_dir_url( __FILE__ );
    $plugin_path = plugin_dir_path( __FILE__ );

    $js_rel  = 'assets/secure-widget.js';
    $css_rel = 'assets/secure-widget.css';

    $js_path  = $plugin_path . $js_rel;
    $css_path = $plugin_path . $css_rel;

    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'MSW: plugin_path=' . $plugin_path );
        error_log( 'MSW: js exists? ' . ( file_exists( $js_path ) ? 'yes' : 'NO' ) . ' -> ' . $js_path );
        error_log( 'MSW: css exists? ' . ( file_exists( $css_path ) ? 'yes' : 'NO' ) . ' -> ' . $css_path );
    }

    if ( file_exists( $js_path ) ) {
        wp_register_script(
            'msw-secure-widget',                    // handle
            $plugin_url . $js_rel,                  // url
            array(),                                // deps
            filemtime( $js_path ),                  // version (cache-bust)
            true                                    // in footer
        );
    }

    if ( file_exists( $css_path ) ) {
        wp_register_style(
            'msw-secure-widget-style',
            $plugin_url . $css_rel,
            array(),
            filemtime( $css_path )
        );
    }
}
add_action( 'wp_enqueue_scripts', 'msw_register_assets' );


/**
 * Helper: convert shortcode attributes into element attributes string.
 * Accepts associative array of attributes, keeps the original keys (including hyphens).
 */
function msw_build_attrs_string( $atts ) {
    $parts = array();
    foreach ( (array) $atts as $key => $value ) {
        // skip empty values to avoid printing empty attributes if you want
        if ( $value === '' || is_null( $value ) ) {
            continue;
        }

        // ensure attribute name is safe (letters, numbers, colon, dot, underscore, dash)
        if ( preg_match( '/^[a-zA-Z0-9\:\.\_\-]+$/', $key ) ) {
            // escape attribute value for HTML
            $parts[] = sprintf( '%s="%s"', esc_attr( $key ), esc_attr( $value ) );
        }
    }
    return implode( ' ', $parts );
}


/**
 * Shortcode: [sba_504_loan_calculator license-key="..."]
 *
 * Ensures assets are enqueued (register first, fallback to direct enqueue if needed).
 */
function msw_render_shortcode( $atts = array() ) {
    $atts = (array) $atts;

    // Ensure assets registered/enqueued when shortcode runs
    // If registered earlier, just enqueue. If not, register+enqueue directly as fallback.
    if ( wp_script_is( 'msw-secure-widget', 'registered' ) ) {
        wp_enqueue_script( 'msw-secure-widget' );
    } else {
        // fallback: register+enqueue directly from plugin folder if file exists
        $js_path = plugin_dir_path( __FILE__ ) . 'assets/secure-widget.js';
        if ( file_exists( $js_path ) ) {
            wp_enqueue_script(
                'msw-secure-widget',
                plugin_dir_url( __FILE__ ) . 'assets/secure-widget.js',
                array(),
                filemtime( $js_path ),
                true
            );
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'MSW: fallback registered+enqueued js' );
            }
        } else {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'MSW: fallback js missing at ' . $js_path );
            }
        }
    }

    if ( wp_style_is( 'msw-secure-widget-style', 'registered' ) ) {
        wp_enqueue_style( 'msw-secure-widget-style' );
    } else {
        $css_path = plugin_dir_path( __FILE__ ) . 'assets/secure-widget.css';
        if ( file_exists( $css_path ) ) {
            wp_enqueue_style(
                'msw-secure-widget-style',
                plugin_dir_url( __FILE__ ) . 'assets/secure-widget.css',
                array(),
                filemtime( $css_path )
            );
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'MSW: fallback registered+enqueued css' );
            }
        } else {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'MSW: fallback css missing at ' . $css_path );
            }
        }
    }

    // Build attributes and output the element
    $attr_string = msw_build_attrs_string( $atts );

    return sprintf( '<sba-504-loan-calculator %s></sba-504-loan-calculator>', $attr_string );
}
add_shortcode( 'sba_504_loan_calculator', 'msw_render_shortcode' );

// optional alias (if you want a shorter name)
add_shortcode( 'my_widget', 'msw_render_shortcode' );
